<?php
session_start();

$configPath = __DIR__ . '/app/config/config.php';
$lockPath = __DIR__ . '/app/config/install.lock';

if (file_exists($configPath) && file_exists($lockPath)) {
    header('Location: app/user/login.php');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dbHost = trim($_POST['db_host'] ?? '');
    $dbName = trim($_POST['db_name'] ?? '');
    $dbUser = trim($_POST['db_user'] ?? '');
    $dbPass = trim($_POST['db_pass'] ?? '');
    $siteName = trim($_POST['site_name'] ?? '');
    $adminEmail = trim($_POST['admin_email'] ?? '');
    $adminPassword = trim($_POST['admin_password'] ?? '');

    if ($dbHost === '' || $dbName === '' || $dbUser === '' || $siteName === '' || $adminEmail === '' || $adminPassword === '') {
        $error = 'Please fill all required fields.';
    } else {
        $mysqli = @new mysqli($dbHost, $dbUser, $dbPass);

        if ($mysqli->connect_error) {
            $error = 'Database connection failed: ' . $mysqli->connect_error;
        } else {
            $dbNameEscaped = $mysqli->real_escape_string($dbName);
            $createDbSql = "CREATE DATABASE IF NOT EXISTS `$dbNameEscaped` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
            if (!$mysqli->query($createDbSql)) {
                $error = 'Failed to create database: ' . $mysqli->error;
            } else {
                if (!$mysqli->select_db($dbNameEscaped)) {
                    $error = 'Failed to select database: ' . $mysqli->error;
                } else {
                    $queries = [];

                    $queries[] = "CREATE TABLE IF NOT EXISTS users (
                        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                        name VARCHAR(100) NOT NULL,
                        email VARCHAR(191) NOT NULL UNIQUE,
                        password_hash VARCHAR(255) NOT NULL,
                        role ENUM('user','admin') NOT NULL DEFAULT 'user',
                        theme_preference ENUM('light','dark','system') NOT NULL DEFAULT 'light',
                        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

                    $queries[] = "CREATE TABLE IF NOT EXISTS tasks (
                        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                        title VARCHAR(150) NOT NULL,
                        description TEXT NULL,
                        reward_amount DECIMAL(10,2) NOT NULL DEFAULT 0,
                        status ENUM('active','inactive','completed') NOT NULL DEFAULT 'active',
                        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

                    $queries[] = "CREATE TABLE IF NOT EXISTS user_tasks (
                        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                        user_id INT UNSIGNED NOT NULL,
                        task_id INT UNSIGNED NOT NULL,
                        status ENUM('assigned','in_progress','submitted','approved','rejected') NOT NULL DEFAULT 'assigned',
                        submitted_at DATETIME NULL,
                        completed_at DATETIME NULL,
                        earning_amount DECIMAL(10,2) NULL,
                        CONSTRAINT fk_user_tasks_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                        CONSTRAINT fk_user_tasks_task FOREIGN KEY (task_id) REFERENCES tasks(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

                    $queries[] = "CREATE TABLE IF NOT EXISTS payouts (
                        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                        user_id INT UNSIGNED NOT NULL,
                        amount DECIMAL(10,2) NOT NULL,
                        status ENUM('pending','paid','failed') NOT NULL DEFAULT 'pending',
                        requested_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        paid_at DATETIME NULL,
                        CONSTRAINT fk_payouts_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

                    $queries[] = "CREATE TABLE IF NOT EXISTS site_settings (
                        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                        setting_key VARCHAR(100) NOT NULL UNIQUE,
                        setting_value TEXT NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

                    foreach ($queries as $sql) {
                        if (!$mysqli->query($sql)) {
                            $error = 'Table creation failed: ' . $mysqli->error;
                            break;
                        }
                    }

                    if ($error === '') {
                        $adminPasswordHash = password_hash($adminPassword, PASSWORD_DEFAULT);

                        $stmt = $mysqli->prepare("INSERT INTO users (name, email, password_hash, role) VALUES (?, ?, ?, 'admin')
                            ON DUPLICATE KEY UPDATE role = VALUES(role)");
                        $adminName = 'Administrator';
                        $stmt->bind_param('sss', $adminName, $adminEmail, $adminPasswordHash);
                        if (!$stmt->execute()) {
                            $error = 'Failed to create admin user: ' . $stmt->error;
                        }
                        $stmt->close();
                    }

                    if ($error === '') {
                        $stmt = $mysqli->prepare("INSERT INTO site_settings (setting_key, setting_value) VALUES 
                            ('site_name', ?),
                            ('monthly_goal', '5000'),
                            ('currency', 'USD')
                        ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
                        $stmt->bind_param('s', $siteName);
                        if (!$stmt->execute()) {
                            $error = 'Failed to save site settings: ' . $stmt->error;
                        }
                        $stmt->close();
                    }

                    if ($error === '') {
                        if (!is_dir(__DIR__ . '/app/config')) {
                            mkdir(__DIR__ . '/app/config', 0775, true);
                        }

                        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
                        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                        $scriptDir = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
                        $base = $scheme . $host . ($scriptDir ? $scriptDir . '/' : '/');

                        $configContent = "<?php\n" .
                            "define('DB_HOST', '" . addslashes($dbHost) . "');\n" .
                            "define('DB_NAME', '" . addslashes($dbName) . "');\n" .
                            "define('DB_USER', '" . addslashes($dbUser) . "');\n" .
                            "define('DB_PASS', '" . addslashes($dbPass) . "');\n" .
                            "define('SITE_NAME', '" . addslashes($siteName) . "');\n" .
                            "define('BASE_URL', '" . addslashes($base) . "');\n\n" .
                            "function db_connection() {\n" .
                            "    static \$mysqli;\n" .
                            "    if (!\$mysqli) {\n" .
                            "        \$mysqli = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);\n" .
                            "        if (\$mysqli->connect_error) {\n" .
                            "            die('Database connection failed');\n" .
                            "        }\n" .
                            "        \$mysqli->set_charset('utf8mb4');\n" .
                            "    }\n" .
                            "    return \$mysqli;\n" .
                            "}\n";

                        if (file_put_contents($configPath, $configContent) === false) {
                            $error = 'Failed to write config file.';
                        } else {
                            file_put_contents($lockPath, 'installed');
                        }
                    }
                }
            }
        }

        if ($error === '') {
            header('Location: app/user/login.php');
            exit;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Install Task Dashboard</title>
    <link rel="stylesheet" href="app/assets/css/style.css">
</head>

<body class="light-theme install-page">
    <div class="install-wrapper">
        <div class="install-card">
            <h1>Task Dashboard Installation</h1>
            <?php if ($error !== ''): ?>
                <div class="alert alert-error">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            <form method="post">
                <div class="form-group">
                    <label for="db_host">Database Host</label>
                    <input type="text" id="db_host" name="db_host" value="localhost" required>
                </div>
                <div class="form-group">
                    <label for="db_name">Database Name</label>
                    <input type="text" id="db_name" name="db_name" required>
                </div>
                <div class="form-group">
                    <label for="db_user">Database User</label>
                    <input type="text" id="db_user" name="db_user" required>
                </div>
                <div class="form-group">
                    <label for="db_pass">Database Password</label>
                    <input type="password" id="db_pass" name="db_pass">
                </div>
                <div class="form-group">
                    <label for="site_name">Site Name</label>
                    <input type="text" id="site_name" name="site_name" required>
                </div>
                <div class="form-group">
                    <label for="admin_email">Admin Email</label>
                    <input type="email" id="admin_email" name="admin_email" required>
                </div>
                <div class="form-group">
                    <label for="admin_password">Admin Password</label>
                    <input type="password" id="admin_password" name="admin_password" required>
                </div>
                <button type="submit" class="btn-primary">Install</button>
            </form>
        </div>
    </div>
</body>

</html>